/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.pdfimport;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import javax.swing.SwingUtilities;
import org.jabref.Globals;
import org.jabref.gui.BasePanel;
import org.jabref.gui.BasePanelMode;
import org.jabref.gui.EntryTypeDialog;
import org.jabref.gui.JabRefFrame;
import org.jabref.gui.externalfiles.DroppedFileHandler;
import org.jabref.gui.externalfiletype.ExternalFileTypes;
import org.jabref.gui.filelist.FileListEntry;
import org.jabref.gui.filelist.FileListTableModel;
import org.jabref.gui.maintable.MainTable;
import org.jabref.gui.undo.UndoableInsertEntry;
import org.jabref.logic.bibtexkeypattern.BibtexKeyGenerator;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.fileformat.PdfContentImporter;
import org.jabref.logic.importer.fileformat.PdfXmpImporter;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.util.UpdateField;
import org.jabref.logic.util.io.FileUtil;
import org.jabref.logic.xmp.XmpUtilShared;
import org.jabref.model.database.KeyCollisionException;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.EntryType;
import org.jabref.pdfimport.ImportDialog;
import org.jabref.pdfimport.PdfFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfImporter {
    private static final Logger LOGGER = LoggerFactory.getLogger(PdfImporter.class);
    private final JabRefFrame frame;
    private final BasePanel panel;
    private final MainTable entryTable;
    private final int dropRow;

    public PdfImporter(JabRefFrame frame, BasePanel panel, MainTable entryTable, int dropRow) {
        this.frame = frame;
        this.panel = panel;
        this.entryTable = entryTable;
        this.dropRow = dropRow;
    }

    public ImportPdfFilesResult importPdfFiles(List<String> fileNames) {
        ArrayList<String> files = new ArrayList<String>(fileNames);
        ArrayList<String> noPdfFiles = new ArrayList<String>();
        for (String file : files) {
            if (PdfFileFilter.accept(file)) continue;
            noPdfFiles.add(file);
        }
        files.removeAll(noPdfFiles);
        List<BibEntry> entries = this.importPdfFilesInternal(files);
        return new ImportPdfFilesResult(noPdfFiles, entries);
    }

    private List<BibEntry> importPdfFilesInternal(List<String> fileNames) {
        if (this.panel == null) {
            return Collections.emptyList();
        }
        ImportDialog importDialog = null;
        boolean doNotShowAgain = false;
        boolean neverShow = Globals.prefs.getBoolean("importAlwaysUsePDFImportStyle");
        int globalChoice = Globals.prefs.getInt("importDefaultPDFimportStyle");
        ArrayList<BibEntry> res = new ArrayList<BibEntry>();
        for (String fileName : fileNames) {
            if (!neverShow && !doNotShowAgain) {
                importDialog = new ImportDialog(this.dropRow >= 0, fileName);
                if (!XmpUtilShared.hasMetadata(Paths.get(fileName, new String[0]), Globals.prefs.getXMPPreferences())) {
                    importDialog.disableXMPChoice();
                }
                importDialog.setLocationRelativeTo(this.frame);
                importDialog.showDialog();
                doNotShowAgain = importDialog.isDoNotShowAgain();
            }
            if (!neverShow && importDialog.getResult() != 0) continue;
            int choice = neverShow ? globalChoice : importDialog.getChoice();
            switch (choice) {
                case 1: {
                    this.doXMPImport(fileName, res);
                    break;
                }
                case 2: {
                    this.doContentImport(fileName, res);
                    break;
                }
                case 0: {
                    this.createNewBlankEntry(fileName).ifPresent(res::add);
                    break;
                }
                case 4: {
                    DroppedFileHandler dfh = new DroppedFileHandler(this.frame, this.panel);
                    if (this.dropRow >= 0) {
                        dfh.linkPdfToEntry(fileName, this.entryTable, this.dropRow);
                        break;
                    }
                    dfh.linkPdfToEntry(fileName, this.entryTable, this.entryTable.getSelectedRow());
                    break;
                }
            }
        }
        return res;
    }

    private void doXMPImport(String fileName, List<BibEntry> res) {
        Path filePath;
        ArrayList<BibEntry> localRes = new ArrayList<BibEntry>();
        PdfXmpImporter importer = new PdfXmpImporter(Globals.prefs.getXMPPreferences());
        ParserResult result = importer.importDatabase(filePath = Paths.get(fileName, new String[0]), Globals.prefs.getDefaultEncoding());
        if (result.hasWarnings()) {
            this.frame.showMessage(result.getErrorMessage());
        }
        localRes.addAll((Collection<BibEntry>)result.getDatabase().getEntries());
        if (localRes.isEmpty()) {
            LOGGER.info("Import failed");
            this.createNewBlankEntry(fileName).ifPresent(res::add);
            return;
        }
        for (BibEntry entry : localRes) {
            this.panel.getDatabase().insertEntry(entry);
            this.panel.markBaseChanged();
            FileListTableModel tm = new FileListTableModel();
            Path toLink = Paths.get(fileName, new String[0]);
            List<Path> dirsS = this.panel.getBibDatabaseContext().getFileDirectoriesAsPaths(Globals.prefs.getFileDirectoryPreferences());
            tm.addEntry(0, new FileListEntry("", FileUtil.shortenFileName(toLink, dirsS).toString(), ExternalFileTypes.getInstance().getExternalFileTypeByName("PDF")));
            entry.setField("file", tm.getStringRepresentation());
            res.add(entry);
        }
    }

    private Optional<BibEntry> createNewBlankEntry(String fileName) {
        Optional<BibEntry> newEntry = this.createNewEntry();
        newEntry.ifPresent(bibEntry -> {
            DroppedFileHandler dfh = new DroppedFileHandler(this.frame, this.panel);
            dfh.linkPdfToEntry(fileName, (BibEntry)bibEntry);
        });
        return newEntry;
    }

    private void doContentImport(String fileName, List<BibEntry> res) {
        Path filePath;
        PdfContentImporter contentImporter = new PdfContentImporter(Globals.prefs.getImportFormatPreferences());
        ParserResult result = contentImporter.importDatabase(filePath = Paths.get(fileName, new String[0]), Globals.prefs.getDefaultEncoding());
        if (result.hasWarnings()) {
            this.frame.showMessage(result.getErrorMessage());
        }
        if (!result.getDatabase().hasEntries()) {
            this.createNewBlankEntry(fileName).ifPresent(res::add);
            return;
        }
        BibEntry entry = (BibEntry)result.getDatabase().getEntries().get(0);
        this.panel.getDatabase().insertEntry(entry);
        this.panel.markBaseChanged();
        new BibtexKeyGenerator(this.panel.getBibDatabaseContext(), Globals.prefs.getBibtexKeyPatternPreferences()).generateAndSetKey(entry);
        DroppedFileHandler dfh = new DroppedFileHandler(this.frame, this.panel);
        dfh.linkPdfToEntry(fileName, entry);
        SwingUtilities.invokeLater(() -> this.panel.highlightEntry(entry));
        if (Globals.prefs.getBoolean("autoOpenForm")) {
            this.panel.showAndEdit(entry);
        }
        res.add(entry);
    }

    private Optional<BibEntry> createNewEntry() {
        EntryTypeDialog etd = new EntryTypeDialog(this.frame);
        etd.setLocationRelativeTo(this.frame);
        etd.setVisible(true);
        EntryType type = etd.getChoice();
        if (type != null) {
            BibEntry bibEntry = new BibEntry(type.getName());
            try {
                this.panel.getDatabase().insertEntry(bibEntry);
                ArrayList<BibEntry> list = new ArrayList<BibEntry>();
                list.add(bibEntry);
                UpdateField.setAutomaticFields(list, true, true, Globals.prefs.getUpdateFieldPreferences());
                this.panel.getUndoManager().addEdit(new UndoableInsertEntry(this.panel.getDatabase(), bibEntry, this.panel));
                this.panel.output(Localization.lang("Added new", new String[0]) + " '" + type.getName().toLowerCase(Locale.ROOT) + "' " + Localization.lang("entry", new String[0]) + ".");
                if (this.panel.getMode() != BasePanelMode.SHOWING_EDITOR) {
                    this.panel.setMode(BasePanelMode.WILL_SHOW_EDITOR);
                }
                SwingUtilities.invokeLater(() -> this.panel.showAndEdit(bibEntry));
                this.panel.markBaseChanged();
                return Optional.of(bibEntry);
            }
            catch (KeyCollisionException ex) {
                LOGGER.info("Key collision occurred", ex);
            }
        }
        return Optional.empty();
    }

    public class ImportPdfFilesResult {
        private final List<String> noPdfFiles;
        private final List<BibEntry> entries;

        public ImportPdfFilesResult(List<String> noPdfFiles, List<BibEntry> entries) {
            this.noPdfFiles = noPdfFiles;
            this.entries = entries;
        }

        public List<String> getNoPdfFiles() {
            return this.noPdfFiles;
        }

        public List<BibEntry> getEntries() {
            return this.entries;
        }
    }
}

